#!/bin/sh

RESULTDIR="."
TOKENPARSE="Saliency Heat Interpolation"
TOKENOK=0

usage(){
  echo "Usage: ./stats.sh -file <logs file> <arguments>"
  echo "If no argument given, the script returns immediatly."
  echo "Values for <arguments> :"
  echo "  -split                print values needed to make a curve."
  echo "  -plot                 display the curves created."
  echo "  -stats                display the curves created."
  echo "  -dir <directory>      store results inside the <directory>, by default it's the current directory."
  echo "  -help, -h             show this help."
}


#----------------------------------------------------------------
# Parse - Compute average and some things interesting...
#----------------------------------------------------------------

parse(){

  VALUES=$(cat $1 | grep $2 | sed -e "s/$2://g")

  #Don't compute if there is empty
  if [ -z "$VALUES" ]
  then
    return 0;
  fi

  MOYENNE=0
  CPT=0
  for j in $VALUES ; do
    MOYENNE=$(($MOYENNE + $j))
    CPT=$(($CPT + 1))
  done
  MOYENNE=$(($MOYENNE / $CPT))

  VARIANCE=0
  for j in $VALUES ; do
    TMP=$(($j - $MOYENNE))
    TMP=$(($TMP * $TMP))
    VARIANCE=$(($VARIANCE + $TMP))
  done
  VARIANCE=$(($VARIANCE / $CPT))

  ECARTTYPE=$(echo "sqrt($VARIANCE)" | bc)

  echo "$2 : µ = $MOYENNE ± $ECARTTYPE µs" >> $FILE;

}

#----------------------------------------------------------------
# GenPlot - Create plot file to display curves
#----------------------------------------------------------------

genPlot(){
  echo -n "#!/usr/bin/gnuplot

  set term png #enhanced monochrome dashed
  set output \"$RESULTDIR/graph.png\"

  set key top left

  set xlabel \"Frame Number\"
  set ylabel \"Time (in µs)\"
  
  plot " > $GPFILE;

}

#----------------------------------------------------------------
# Makecurve - Print results in separatly files to make curves
#----------------------------------------------------------------

makecurve(){

  VALUES=$(cat $1 | grep $2 | sed -e "s/$2://g")

  #Don't compute if there is empty
  if [ -z "$VALUES" ]
  then
    return 0;
  fi

  FILE=$( mktemp -p $RESULTDIR $2.XXXXXX )

  CPT=0
  for j in $VALUES ; do
    echo "$CPT $j" >> $FILE
    CPT=$(($CPT + 1))
  done

  echo "Results stored in $FILE !"

  if [ ! -z $SHOW ]
  then
    echo "\"$FILE\" title \"$2\" with lines, \\" >> $GPFILE;
  fi

  TOKENOK=$(($TOKENOK + 1))
}

makecurvesum(){

  VALUES=$(cat $1)

  for j in $TOKENPARSE ; do
    VALUES=$(echo $VALUES | sed -e "s/$j://g")
  done

  #Don't compute if there is empty
  if [ -z "$VALUES" ]
  then
    return 0;
  fi

  FILE=$( mktemp -p $RESULTDIR Sum.XXXXXX )

  CPT=0
  TMP=0
  for j in $VALUES ; do
    TMP=$(($TMP + $j))
    CPT=$(($CPT + 1))
    if [ $(($CPT % $2 )) == 0 ]
    then
      echo "$(($CPT / $2)) $TMP" >> $FILE
      TMP=0
    fi
  done

  echo "Results stored in $FILE !"

  if [ ! -z $SHOW ]
  then
    echo "\"$FILE\" title \"Total Time\" with lines, \\" >> $GPFILE;
  fi
  
}

#----------------------------------------------------------------
# Curve - Print results to make curves
#----------------------------------------------------------------

curve(){
  #create plot configuration file
  if [ ! -z $SHOW ]
  then
    genPlot;
  fi

  #parse file with token
  for j in $TOKENPARSE ; do
    makecurve $1 $j;
  done

  #uncomment if you want to display the time spend to compute one frame.
  #makecurvesum $1 $TOKENOK;
  
}

#----------------------------------------------------------------
# Main
#----------------------------------------------------------------

#parse parameters
while [[ $# > 0 ]]
do
  key="$1"

  case $key in
    "-help"|"-h")
      usage;
      exit;
    ;;
    "-file")
      GLOBALFILE="$2"
      shift
    ;;
    "-split")
      CURVE="OK"
    ;;
    "-plot")
      SHOW="OK"
    ;;
    "-stats")
      STATS="OK"
    ;;
    "-dir")
      RESULTDIR="$2"
      if [ ! -e $RESULTDIR ]
      then
        mkdir $RESULTDIR;
      fi
      shift
    ;;
    *)
      usage;
      exit;
    ;;
  esac
shift 
done

GPFILE=$( mktemp -p $RESULTDIR ConfGnuPlot.XXXXXX )

if [ -z $GLOBALFILE ] || [ ! -e $GLOBALFILE ]
then
  echo "Logs file is empty or doesn't exist !"
  usage;
  exit;
fi

#if we need to compute curve
if [ ! -z $CURVE ]
then
    curve $GLOBALFILE;
    echo "GnuPlot Configuration File created in $GPFILE !"
    gnuplot $GPFILE;
fi

if [ ! -z $STATS ]
then
  FILE=$( mktemp -p $RESULTDIR Stats.XXXXXX )
  #parse file with token
  for j in $TOKENPARSE ; do
    parse $GLOBALFILE $j
  done
  echo "Stats stored in $FILE !"
fi
